
// LB+ guard with mapping + persistence (does not touch styles/markup)
(function(){
  const log = (...a) => console.log('[LB2]', ...a);
  const $ = id => document.getElementById(id);
  const safe = v => (typeof v === 'number' && isFinite(v)) ? v : 0;

  const KNOWN = ['FR','HB','HL','NR','NG','RF'];
  function mapHouse(code, name){
    const c = (code||'').toUpperCase().trim();
    const n = (name||'').toUpperCase();
    if (KNOWN.includes(c)) return c;
    // prefix or common variants
    if (c.startsWith('FR') || n.includes('FREY')) return 'FR';
    if (c.startsWith('HB') || n.includes('HALB')) return 'HB';
    if (c.startsWith('HL') || n.includes('HILL')) return 'HL';
    if (c.startsWith('NR') || n.includes('NGAR')) return 'NR';
    if (c.startsWith('NG') || n.includes('NGAT')) return 'NG';
    if (c.startsWith('RF') || n.includes('RUTH')) return 'RF';
    return c || 'FR'; // default to FR so text is visible (has a background)
  }

  function buildRow(r){
    const code = mapHouse(r.house_code, r.house_name);
    return `
      <tr class="house-row ${code}">
        <td class="col-code"><strong>${r.house_code ?? ''}</strong></td>
        <td class="col-house" style="text-align:left;font-weight:800">${r.house_name ?? ''}</td>

        <td class="gap" aria-hidden="true"></td>

        <td class="first-in-group">${safe(r.jun_comp)}</td>
        <td>${safe(r.jun_part)}</td>
        <td class="emph">${safe(r.jun_total)}</td>
        <td>${r.jun_rank ?? ''}</td>

        <td class="gap" aria-hidden="true"></td>

        <td class="first-in-group">${safe(r.int_comp)}</td>
        <td>${safe(r.int_part)}</td>
        <td class="emph">${safe(r.int_total)}</td>
        <td>${r.int_rank ?? ''}</td>

        <td class="gap" aria-hidden="true"></td>

        <td class="first-in-group">${safe(r.sen_comp)}</td>
        <td>${safe(r.sen_part)}</td>
        <td class="emph">${safe(r.sen_total)}</td>
        <td>${r.sen_rank ?? ''}</td>

        <td class="gap" aria-hidden="true"></td>

        <td class="first-in-group emph">${safe(r.tot_comp)}</td>
        <td class="emph">${safe(r.tot_part)}</td>
        <td class="emph" style="font-size:20px">${safe(r.overall)}</td>
        <td>${r.overall_rank ?? ''}</td>
      </tr>
    `;
  }

  async function getRows(){
    const res = await fetch('event_api.php?action=overall_grid', { headers:{'Accept':'application/json'} });
    const txt = await res.text();
    let data;
    try { data = JSON.parse(txt); } catch(e){ log('JSON parse error', e, txt.slice(0,200)); return []; }
    if (!data || data.ok !== true) { log('endpoint not ok', data); return []; }
    const arr = Array.isArray(data.rows) ? data.rows.slice() : [];
    arr.sort((a,b)=> safe(b.overall) - safe(a.overall));
    return arr;
  }

  function render(rows){
    const body = $('lbBody');
    if (!body) { log('no #lbBody'); return; }
    if (!Array.isArray(rows) || rows.length === 0){
      body.innerHTML = '<tr><td colspan="22" style="text-align:center;padding:16px">No data yet</td></tr>';
      return;
    }
    const html = rows.map(buildRow).join('');
    body.innerHTML = html;
    log('rendered', rows.length, 'rows');
  }

  async function run(){
    const rows = await getRows();
    render(rows);
  }

  // Persist rows if some other script tries to wipe them
  function guard(){
    const body = $('lbBody');
    if (!body) return;
    const mo = new MutationObserver((muts) => {
      for (const m of muts){
        if (m.type === 'childList' && body.children.length === 0){
          run().catch(console.error);
          break;
        }
      }
    });
    mo.observe(body, { childList: true });
  }

  // Run after window load to avoid other scripts clearing later
  if (document.readyState === 'complete') setTimeout(run, 0);
  else window.addEventListener('load', () => setTimeout(run, 0), { once: true });

  // Hook refresh button if present
  const btn = document.getElementById('btnRefresh');
  if (btn) btn.addEventListener('click', () => run().catch(console.error));

  guard();
})();
