<?php
// Page shell
$page_title = "TBC Athletics — Event Entry";
$active = 'entry';
$container_class = 'container container-wide'; // widen page
include __DIR__ . '/partials/header.php';
?>

<style>
  /* ---------- Event cards ---------- */
  .event-grid{ display:grid; grid-template-columns: repeat(auto-fill, minmax(220px,1fr)); gap:14px; }
  .event-card{ cursor:pointer; user-select:none; display:flex; align-items:center; gap:10px; transition:transform .08s, box-shadow .08s; }
  .event-card:hover{ transform:translateY(-1px); box-shadow:0 10px 20px rgba(0,0,0,.07) }
  .event-card .emoji{ font-size:20px; opacity:.9 }
  .event-card .title{ font-weight:800 }
  .event-card .sub{ font-size:12px; opacity:.7; margin-top:2px }

  /* ---------- Panel ---------- */
  .entry-panel{ display:none; margin-top:14px }
  .entry-panel.active{ display:block }
  .tabs{ display:flex; gap:8px; margin:10px 0 6px }
  .tabbtn{ border:0; padding:8px 12px; border-radius:10px; background:#e6ecf3; font-weight:800; cursor:pointer }
  .tabbtn.active{ background:#cfd8e3 }

  .rows{ display:grid; gap:12px }

  /* Row: place | block-of-ties */
  .row{
    display:grid; grid-template-columns: 56px 1fr; /* slimmer place column */
    gap:12px; align-items:start;
    background:#f7f9fc; border:1px solid #dee5ef; border-radius:12px; padding:12px;
  }
  .row.place-1{ background:linear-gradient(0deg,#fff8d8,#fff7c2); border-color:#e5cf7a }   /* gold */
  .row.place-2{ background:linear-gradient(0deg,#f4f6f9,#eef1f6); border-color:#cfd6df }   /* silver */
  .row.place-3{ background:linear-gradient(0deg,#fbe8d6,#f8dfc5); border-color:#e6b086 }   /* bronze */

  .row .place{ font-weight:900; text-align:center; padding-top:8px; white-space:nowrap }

  .multi{ display:grid; gap:12px }

  /* One tie entry (left -> right):
     [Name (search)] [Action] [Perf] [Year] [House] */
  .selset{
    display:grid;
    grid-template-columns: minmax(420px, 1fr) 100px 220px 140px 220px; /* name | button | perf | year | house */
    gap:10px; align-items:center;
  }
  .student-input, .perf-input{
    padding:12px 14px; border-radius:10px; border:1px solid #cbd5e1; background:#fff; font-size:14px; width:100%;
  }
  .act-btn{
    width:100%; padding:12px 14px; border-radius:10px;
    border:1px solid #cbd5e1; background:#fff; font-weight:800; cursor:pointer;
  }
  .act-btn:hover{ background:#eef3f8 }

  .info-box{
    padding:12px 14px; border-radius:10px; border:1px solid #cbd5e1; background:#fff;
    font-size:14px; font-weight:800; text-align:center; text-transform:uppercase;
  }
  .info-box.placeholder{ color:#7b8794; background:#eef3f8; font-weight:600 }
  /* force widths */
  .info-box[data-role="year"]{ width:140px !important; min-width:140px !important; }
  .info-box[data-role="house"]{ width:220px !important; min-width:220px !important; }

  /* House colours */
  .info-box.FR{ background:var(--FR); color:#fff; border-color:transparent }
  .info-box.HB{ background:var(--HB); color:#fff; border-color:transparent }
  .info-box.HL{ background:var(--HL); color:#111; border-color:transparent }
  .info-box.NR{ background:var(--NR); color:#fff; border-color:transparent }
  .info-box.NG{ background:#fff; color:#111 }
  .info-box.RF{ background:var(--RF); color:#fff; border-color:transparent }

  .rec-flag{ font-weight:900; color:#047857; display:none; margin-left:8px }
  .rec-flag.show{ display:inline }

  .savebar{ display:flex; justify-content:flex-end; gap:8px; margin-top:10px }
  .button.secondary{ background:#e6ecf3; color:#0b2230 }
</style>

<section class="card">
  <div class="flex justify-between items-center">
    <h2>Event Entry</h2>
  </div>

  <div class="event-grid" id="eventGrid"></div>

  <div id="entryPanel" class="card entry-panel">
    <div class="flex justify-between items-center">
      <div>
        <h3 id="entryTitle" style="margin:0"></h3>
        <div id="entryNote" class="note"></div>
      </div>
      <div class="savebar">
        <button class="button secondary" id="closePanel">Close</button>
        <button class="button" id="saveAll">Save Results</button>
      </div>
    </div>

    <div class="tabs">
      <button class="tabbtn active" data-group="junior" id="tabJunior">Junior</button>
      <button class="tabbtn" data-group="intermediate" id="tabInter">Intermediate</button>
      <button class="tabbtn" data-group="senior" id="tabSenior">Senior</button>
    </div>

    <div id="rowsContainer"></div>
  </div>
</section>

<script>
const API_BASE = "<?= url('event_api.php') ?>";

/* ---------- Events ---------- */
const EVENTS = [
  {id:'100m',   label:'100m',   kind:'track'},
  {id:'200m',   label:'200m',   kind:'track'},
  {id:'300_400',label:'300m / 400m', kind:'track'},
  {id:'800m',   label:'800m',   kind:'track'},
  {id:'1500m',  label:'1500m',  kind:'track'},
  {id:'3000m',  label:'3000m',  kind:'track'},
  {id:'LJ',     label:'Long Jump',   kind:'field'},
  {id:'TJ',     label:'Triple Jump', kind:'field'},
  {id:'HJ',     label:'High Jump',   kind:'field'},
  {id:'SP',     label:'Shot Put',    kind:'field'},
  {id:'DI',     label:'Discus',      kind:'field'},
  {id:'JV',     label:'Javelin',     kind:'field'},
  {id:'4x100',  label:'4 × 100m',    kind:'track'}
];

/* ---------- Students ---------- */
const StudentsByGroup = { junior:[], intermediate:[], senior:[] };

async function loadStudents(){
  const res = await fetch(API_BASE + '?action=students');
  const j = await res.json();
  if(!j.ok) throw new Error(j.error||'students');
  ['junior','intermediate','senior'].forEach(g => StudentsByGroup[g] = j[g]||[]);
  buildDatalists();
}

/* datalists so search is inside the input */
function buildDatalists(){
  const root = document.body;
  ['junior','intermediate','senior'].forEach(g=>{
    const id = `dl-${g}`;
    if(document.getElementById(id)) return;
    const dl = document.createElement('datalist');
    dl.id = id;
    const arr = StudentsByGroup[g] || [];
    dl.innerHTML = arr.map(p=>{
      const label = `${p.last_name}, ${p.first_name}`;
      return `<option value="${label}" data-code="${p.student_code}" data-year="${p.year_level||''}" data-house="${p.house_code||''}"></option>`;
    }).join('');
    root.appendChild(dl);
  });
}
function getOptionFromDatalist(id, value){
  const dl = document.getElementById(id);
  if(!dl) return null;
  return Array.from(dl.options).find(o => o.value === value) || null;
}

/* ---------- Cards ---------- */
function renderEventCards(){
  const grid = document.getElementById('eventGrid');
  grid.innerHTML = EVENTS.map(ev => `
    <div class="card event-card" data-ev="${ev.id}" data-kind="${ev.kind}">
      <div class="emoji">&#127939</div>
      <div>
        <div class="title">${ev.label}</div>
        <div class="sub">${ev.kind==='track' ? 'Track' : 'Field'} event</div>
      </div>
    </div>
  `).join('');
  grid.querySelectorAll('.event-card').forEach(card=>{
    card.addEventListener('click', ()=>openPanel(card.dataset.ev));
  });
}

/* ---------- Panel ---------- */
let CurrentEvent = null;
function EventsKind(){ return (CurrentEvent?.id==='300_400') ? 'track' : (CurrentEvent?.kind||'track'); }
function openPanel(evId){
  CurrentEvent = EVENTS.find(e=>e.id===evId);
  if(!CurrentEvent) return;
  document.getElementById('entryTitle').textContent = CurrentEvent.label;
  document.getElementById('entryNote').textContent =
    (evId==='300_400') ? 'Note: Juniors run 300m; Intermediate/Senior run 400m.' :
    (CurrentEvent.kind==='track' ? 'Enter times as mm:ss.ss or ss.ss' : 'Enter distances in meters (e.g., 5.86)');
  document.querySelectorAll('.tabbtn').forEach(b=>b.classList.remove('active'));
  document.getElementById('tabJunior').classList.add('active');
  renderRows('junior');
  document.getElementById('entryPanel').classList.add('active');
}
document.getElementById('closePanel').addEventListener('click', ()=>document.getElementById('entryPanel').classList.remove('active'));
document.querySelectorAll('.tabbtn').forEach(btn=>{
  btn.addEventListener('click', ()=>{
    document.querySelectorAll('.tabbtn').forEach(b=>b.classList.remove('active'));
    btn.classList.add('active');
    renderRows(btn.dataset.group);
  });
});

/* ---------- Rows with ties ---------- */
function ordinal(n){ const s=["th","st","nd","rd"], v=n%100; return n+(s[(v-20)%10]||s[v]||s[0]); }

function renderRows(group){
  const cont = document.getElementById('rowsContainer');
  cont.innerHTML = `
    <div class="rows" data-group="${group}">
      ${[1,2,3,4,5].map(place => rowHtml(place, group)).join('')}
    </div>
  `;
  // create first selector in each row + bind row-level click handler once
  document.querySelectorAll(`.row[data-group="${group}"]`).forEach(r=>{
    addTieEntry(r, group, true);
    if (!r.dataset.bound) {
      r.addEventListener('click', onRowClick);
      r.dataset.bound = '1';
    }
  });
  recomputePlaces(group);
  applyPlaceColours(group);
}

function rowHtml(place, group){
  return `
  <div class="row" data-base="${place}" data-group="${group}">
    <div class="place"></div>
    <div class="multi" data-role="multi"></div>
  </div>`;
}

/* Build one tie-entry block */
function addTieEntry(row, group, first=false){
  const dl = `dl-${group}`;
  const wrap = row.querySelector('[data-role="multi"]');

  const el = document.createElement('div');
  el.className = 'selset';
  el.innerHTML = `
    <input class="student-input" list="${dl}" placeholder="Search & select student…" data-code="">
    <button type="button" class="act-btn" data-mode="tie">+ Tie</button>
    <div style="display:flex;align-items:center;gap:8px">
      <input class="perf-input" data-role="perf" placeholder="${EventsKind()==='track'?'mm:ss.ss or ss.ss':'meters'}">
      <span class="rec-flag" data-role="record">New Record!</span>
    </div>
    <div class="info-box placeholder" data-role="year">Year</div>
    <div class="info-box placeholder" data-role="house">House</div>
  `;
  wrap.appendChild(el);

  const nameInput = el.querySelector('.student-input');
  const yearBox   = el.querySelector('[data-role="year"]');
  const houseBox  = el.querySelector('[data-role="house"]');
  const perf      = el.querySelector('.perf-input');
  const recFlag   = el.querySelector('[data-role="record"]');

  // student chosen -> update year/house chips
  nameInput.addEventListener('change', ()=>{
    const opt = getOptionFromDatalist(dl, nameInput.value);
    const yr  = opt?.dataset?.year || '';
    const hs  = opt?.dataset?.house || '';
    const code= opt?.dataset?.code || '';
    nameInput.dataset.code = code;

    yearBox.textContent = yr ? `Y${yr}` : 'Year';
    yearBox.className   = 'info-box ' + (yr ? '' : 'placeholder');
    houseBox.textContent= hs || 'House';
    houseBox.className  = 'info-box ' + (hs ? hs : 'placeholder');

    recomputePlaces(row.dataset.group);
    applyPlaceColours(row.dataset.group);
  });

  // record check per entry
  perf.addEventListener('change', async ()=>{
    recFlag.classList.remove('show');
    const val = parsePerformance(perf.value, EventsKind()); if(val==null) return;
    const group = row.dataset.group;
    const effectiveEvent = (CurrentEvent.id==='300_400') ? (group==='junior'?'300m':'400m') : CurrentEvent.id;
    try{
      const res = await fetch(`${API_BASE}?action=check_record&event=${encodeURIComponent(effectiveEvent)}&group=${encodeURIComponent(group)}&kind=${EventsKind()}&value=${encodeURIComponent(val)}`);
      const j = await res.json(); if(j.ok && j.is_record) recFlag.classList.add('show');
    }catch(_){}
  });

  refreshActionButtons(row);
}

/* Single delegated handler for all +Tie/Remove in a row */
function onRowClick(e){
  const btn = e.target.closest('.act-btn');
  if (!btn || !this.contains(btn)) return;
  const row = this;
  const mode = btn.dataset.mode;
  if (mode === 'tie') {
    addTieEntry(row, row.dataset.group, false);
  } else if (mode === 'remove') {
    const set = btn.closest('.selset');
    const wrap = row.querySelector('[data-role="multi"]');
    set.remove();
    // Always keep at least one entry
    if (wrap.querySelectorAll('.selset').length === 0) addTieEntry(row, row.dataset.group, true);
  }
  recomputePlaces(row.dataset.group);
  applyPlaceColours(row.dataset.group);
  refreshActionButtons(row);
}

/* Ensure only LAST entry shows + Tie; others show Remove */
function refreshActionButtons(row){
  const sets = Array.from(row.querySelectorAll('.selset'));
  sets.forEach((s,i)=>{
    const btn = s.querySelector('.act-btn');
    if (!btn) return;
    if (i === sets.length-1) {
      btn.textContent = '+ Tie';
      btn.dataset.mode = 'tie';
    } else {
      btn.textContent = 'Remove';
      btn.dataset.mode = 'remove';
    }
  });
}

/* Places with ties (dense progression) */
function recomputePlaces(group){
  const rows = Array.from(document.querySelectorAll(`.row[data-group="${group}"]`));
  let pos = 1;
  rows.forEach(r=>{
    r.dataset.placecalc = pos;
    r.querySelector('.place').textContent = ordinal(pos);
    const selected = Array.from(r.querySelectorAll('.student-input'))
      .filter(inp => (inp.dataset.code||'').trim() !== '').length;
    pos += Math.max(selected, 1);
  });
}

/* Medal backgrounds for current 1st/2nd/3rd rows */
function applyPlaceColours(group){
  const rows = Array.from(document.querySelectorAll(`.row[data-group="${group}"]`));
  rows.forEach(r=> r.classList.remove('place-1','place-2','place-3'));
  const byPlace = [...rows].sort((a,b)=>(+a.dataset.placecalc||+a.dataset.base) - (+b.dataset.placecalc||+b.dataset.base));
  byPlace.forEach(r=>{
    const p = parseInt(r.dataset.placecalc||r.dataset.base,10);
    if (p===1) r.classList.add('place-1');
    if (p===2) r.classList.add('place-2');
    if (p===3) r.classList.add('place-3');
  });
}

/* Save */
document.getElementById('saveAll').addEventListener('click', async ()=>{
  if(!CurrentEvent) return;
  const payload = { event: CurrentEvent.id, entries: [] };

  document.querySelectorAll('#rowsContainer .row').forEach(r=>{
    const group = r.dataset.group;
    const place = parseInt(r.dataset.placecalc || r.dataset.base, 10);
    r.querySelectorAll('.selset').forEach(set=>{
      const code = (set.querySelector('.student-input')?.dataset?.code || '').trim();
      const perfStr = (set.querySelector('.perf-input')?.value || '').trim();
      if(!code || !perfStr) return;
      const perfVal = parsePerformance(perfStr, EventsKind());
      payload.entries.push({ group, place, student_code: code, performance: perfStr, perf_value: perfVal });
    });
  });

  if(payload.entries.length===0){ alert('Nothing to save yet.'); return; }

  const res = await fetch(API_BASE + '?action=save_results', {
    method:'POST', headers:{'Content-Type':'application/json'},
    body: JSON.stringify(payload)
  });
  const j = await res.json();
  if(!j.ok) return alert(j.error || 'Save failed');
  alert('Saved!');
});

/* Utils */
function parsePerformance(str, kind){
  if(!str) return null;
  str = String(str).trim();
  if(kind==='track'){
    if(str.includes(':')){
      const [mm, ss] = str.split(':');
      const secs = (parseInt(mm,10)||0) * 60 + (parseFloat(ss)||0);
      return isNaN(secs) ? null : secs;
    } else {
      const secs = parseFloat(str);
      return isNaN(secs) ? null : secs;
    }
  } else {
    const m = parseFloat(str);
    return isNaN(m) ? null : m;
  }
}

/* Boot */
(async function(){
  try{
    renderEventCards();
    await loadStudents();
  }catch(e){
    console.error(e);
    alert('Failed to load students: ' + (e.message||e));
  }
})();
</script>

<?php include __DIR__ . '/partials/footer.php'; ?>
