<?php
/* ================================================================
   ALWAYS JSON — install safety net BEFORE any includes
================================================================ */
header('Content-Type: application/json; charset=utf-8');
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Start buffer immediately so even include-time output is captured
ob_start();

// Convert PHP warnings/notices to JSON (no HTML)
set_error_handler(function($severity, $message, $file, $line) {
  http_response_code(500);
  echo json_encode(['ok' => false, 'error' => $message, 'file' => $file, 'line' => $line]);
  return true;
});

// Convert uncaught exceptions to JSON
set_exception_handler(function($e) {
  http_response_code(500);
  echo json_encode(['ok' => false, 'error' => $e->getMessage()]);
});

// On shutdown, if anything was printed, make sure it’s JSON
register_shutdown_function(function() {
  $buf = ob_get_contents(); // read buffer
  if ($buf === false) { return; }
  ob_end_clean(); // clear it; we will re-emit below

  if ($buf === '') {
    // nothing printed — OK
    return;
  }

  $trim = ltrim($buf);
  if ($trim !== '' && (isset($trim[0]) && ($trim[0] === '{' || $trim[0] === '['))) {
    // It already looks like JSON; pass it through
    echo $buf;
  } else {
    // It was HTML/notice text — wrap as JSON so the frontend can parse it
    http_response_code(500);
    echo json_encode(['ok' => false, 'error' => 'PHP output before JSON', 'details' => $buf]);
  }
});

/* ================================================================
   Includes (after safety net)
================================================================ */
require_once __DIR__ . '/../lib/db.php';

// Create PDO with guard
try {
  $pdo = get_pdo();
} catch (Throwable $e) {
  echo json_encode(['ok'=>false,'error'=>'DB connection failed','details'=>$e->getMessage()]);
  exit;
}

/* ================================================================
   Helpers
================================================================ */
function json_ok($arr = []){ echo json_encode(['ok'=>true] + $arr); exit; }
function json_err($msg, $code=500){ http_response_code($code); echo json_encode(['ok'=>false,'error'=>$msg]); exit; }

$action = isset($_GET['action']) ? $_GET['action'] : '';

/* ================================================================
   1) students — grouped by Junior/Intermediate/Senior
================================================================ */
if ($action === 'students'){
  $stmt = $pdo->query("
    SELECT student_code, first_name, last_name, year_level, house_code
    FROM students
    ORDER BY last_name, first_name
  ");

  $jun = []; $int = []; $sen = []; $all = [];
  while ($r = $stmt->fetch(PDO::FETCH_ASSOC)) {
    $yl = isset($r['year_level']) ? (int)$r['year_level'] : null;
    if ($yl === 9 || $yl === 10) { $jun[] = $r; }
    elseif ($yl === 11)        { $int[] = $r; }
    elseif ($yl === 12 || $yl === 13) { $sen[] = $r; }
    $all[] = $r;
  }

  if (!empty($_GET['debug'])) {
    $yr_counts = [];
    foreach ([9,10,11,12,13] as $y) {
      $c = 0; foreach ($all as $row) { if ((int)$row['year_level'] === $y) $c++; }
      $yr_counts[$y] = $c;
    }
    json_ok(['junior'=>$jun,'intermediate'=>$int,'senior'=>$sen,'meta'=>[
      'total_in_students'=>count($all),'yr_counts'=>$yr_counts
    ]]);
  }

  json_ok(['junior'=>$jun, 'intermediate'=>$int, 'senior'=>$sen]);
}

/* ================================================================
   2) check_record — inline record flag on entry rows
================================================================ */
if ($action === 'check_record'){
  $event = isset($_GET['event']) ? $_GET['event'] : '';
  $group = isset($_GET['group']) ? $_GET['group'] : '';
  $kind  = isset($_GET['kind'])  ? $_GET['kind']  : 'track';
  $val   = isset($_GET['value']) ? (float)$_GET['value'] : null;
  if ($event==='' || $group==='' || $val===null) json_err('bad params', 400);

  $q = $pdo->prepare("SELECT value, unit FROM records WHERE event_code = ? AND age_group = ?");
  $q->execute([$event, $group]);
  $rec = $q->fetch(PDO::FETCH_ASSOC);

  if (!$rec){ json_ok(['is_record'=>false, 'no_record'=>true]); }

  $isRecord = ($kind==='track') ? ($val < (float)$rec['value']) : ($val > (float)$rec['value']);
  json_ok(['is_record'=>$isRecord, 'record_value'=>(float)$rec['value']]);
}

/* ================================================================
   3) save_results — competitive results from entry page
================================================================ */
if ($action === 'save_results'){
  $data = json_decode(file_get_contents('php://input'), true);
  if (!$data || !isset($data['event']) || !isset($data['entries'])) json_err('bad payload', 400);

  $event   = $data['event'];
  $entries = $data['entries'];

  $ins = $pdo->prepare("
    INSERT INTO competitive_results
      (event_code, age_group, student_code, performance_text, perf_value, created_at)
    VALUES (?,?,?,?,?,NOW())
  ");

  foreach($entries as $e){
    if (empty($e['student_code']) || empty($e['group']) || empty($e['performance'])) continue;
    $ins->execute([$event, $e['group'], $e['student_code'], $e['performance'], (isset($e['perf_value']) ? $e['perf_value'] : null)]);
  }
  json_ok();
}

/* ================================================================
   4) overall_grid — podium + leaderboard data (schema-tolerant)
================================================================ */
if ($action === 'overall_grid'){
  // 0) Houses
  $houses = [];
  try {
    $h = $pdo->query("SELECT house_code, house_name FROM houses");
    while ($r = $h->fetch(PDO::FETCH_ASSOC)) $houses[$r['house_code']] = $r['house_name'];
  } catch (Throwable $e) { /* fallback below */ }
  if (!$houses) {
    $houses = ['FR'=>'Freyberg','HB'=>'Halberg','HL'=>'Hillary','NR'=>'Ngārimu','NG'=>'Ngata','RF'=>'Rutherford'];
  }

  // Base rows per house
  $rows = [];
  foreach ($houses as $code=>$name) {
    $rows[$code] = [
      'house_code'=>$code, 'house_name'=>$name,
      'jun_comp'=>0,'jun_part'=>0,'jun_total'=>0,'jun_rank'=>null,
      'int_comp'=>0,'int_part'=>0,'int_total'=>0,'int_rank'=>null,
      'sen_comp'=>0,'sen_part'=>0,'sen_total'=>0,'sen_rank'=>null,
      'tot_comp'=>0,'tot_part'=>0,'overall'=>0,'overall_rank'=>null,
    ];
  }

  // --- 1) Work out column names present in your DB ---
  $crCols = $pdo->query("SHOW COLUMNS FROM competitive_results")->fetchAll(PDO::FETCH_COLUMN, 0);
  $stCols = $pdo->query("SHOW COLUMNS FROM students")->fetchAll(PDO::FETCH_COLUMN, 0);

  $pick = function(array $want, array $have){
    foreach ($want as $w) if (in_array($w, $have, true)) return $w;
    return null;
  };

  $cr_student = $pick(['student_code','student_id','student','sid','stu_code'], $crCols);
  $cr_age     = $pick(['age_group','group','age','division','grade'], $crCols);
  $cr_event   = $pick(['event_code','event','event_id','eventid','code','evt'], $crCols);
  $cr_perf    = $pick(['perf_value','performance','perf','value','result','mark','time','distance'], $crCols);

  $st_student = $pick(['student_code','student_id','id','code'], $stCols);
  $st_house   = $pick(['house_code','house','houseid','house_id'], $stCols);

  if (!$cr_student || !$cr_age || !$cr_event || !$cr_perf || !$st_student || !$st_house) {
    // Missing essentials: return zeros so the page still loads
    json_ok(['rows'=>array_values($rows)]);
  }

  // --- 2) Query results joined to students for house ---
  $sql = "
    SELECT cr.`$cr_event`   AS event_key,
           cr.`$cr_age`     AS age_group,
           cr.`$cr_perf`    AS perf_value,
           s.`$st_house`    AS house_code
    FROM competitive_results cr
    JOIN students s ON s.`$st_student` = cr.`$cr_student`
    WHERE cr.`$cr_perf` IS NOT NULL AND s.`$st_house` IS NOT NULL
  ";
  $res = $pdo->query($sql);

  // --- 3) Group rows by age & event ---
  $groups = [];
  $canon_age = function($v){
    $x = strtolower(trim((string)$v));
    $x = preg_replace('/[^a-z]/','',$x);
    if (in_array($x, ['j','jr','jun','junior','juniors'])) return 'junior';
    if (in_array($x, ['i','int','inter','intermediate','intermediates','mid'])) return 'intermediate';
    if (in_array($x, ['s','snr','sen','senior','seniors'])) return 'senior';
    return null;
  };
  while ($r = $res->fetch(PDO::FETCH_ASSOC)) {
    $g = $canon_age($r['age_group']);
    if ($g === null) continue;
    $e = (string)$r['event_key'];
    if (!isset($groups[$g])) $groups[$g] = [];
    if (!isset($groups[$g][$e])) $groups[$g][$e] = [];
    $groups[$g][$e][] = $r;
  }

  // --- 4) Score: 1st=10, 2nd=6, 3rd=4 (dense ranking), field vs track sort ---
  $fieldSet = [
    'LJ'=>1,'TJ'=>1,'HJ'=>1,'SP'=>1,'DI'=>1,'JV'=>1,
    'LONGJUMP'=>1,'TRIPLEJUMP'=>1,'HIGHJUMP'=>1,'SHOTPUT'=>1,'DISCUS'=>1,'JAVELIN'=>1
  ];
  foreach ($groups as $age => $events) {
    foreach ($events as $ev => $list) {
      $evKey = strtoupper((string)$ev);
      $evKeyLetters = preg_replace('/[^A-Z]/', '', $evKey);
      $isField = isset($fieldSet[$evKey]) || isset($fieldSet[$evKeyLetters]) ||
                 preg_match('/(JUMP|PUT|DISC|JAV|THROW)/', $evKey);

      usort($list, function($a,$b) use ($isField){
        if ($a['perf_value'] == $b['perf_value']) return 0;
        return $isField
          ? (($a['perf_value'] > $b['perf_value']) ? -1 : 1)  // field: higher is better
          : (($a['perf_value'] < $b['perf_value']) ? -1 : 1); // track: lower is better
      });

      // dense places
      $points = [1=>10,2=>6,3=>4];
      $i = 0; $n = count($list); $place = 1;
      while ($i < $n && $place <= 3) {
        $start = $i;
        $v = (string)$list[$i]['perf_value'];
        while ($i < $n && (string)$list[$i]['perf_value'] === $v) $i++;
        $len = $i - $start;
        if (isset($points[$place])) {
          $pts = $points[$place];
          for ($k = $start; $k < $i; $k++) {
            $hc = $list[$k]['house_code'];
            if (!isset($rows[$hc])) continue;
            if ($age === 'junior')           $rows[$hc]['jun_comp'] += $pts;
            elseif ($age === 'intermediate') $rows[$hc]['int_comp'] += $pts;
            elseif ($age === 'senior')       $rows[$hc]['sen_comp'] += $pts;
          }
        }
        $place += $len;
      }
    }
  }

  // --- 5) Totals & overall ---
  foreach ($rows as &$r) {
    $r['jun_total'] = $r['jun_comp'] + $r['jun_part'];
    $r['int_total'] = $r['int_comp'] + $r['int_part'];
    $r['sen_total'] = $r['sen_comp'] + $r['sen_part'];
    $r['tot_comp']  = $r['jun_comp'] + $r['int_comp'] + $r['sen_comp'];
    $r['tot_part']  = $r['jun_part'] + $r['int_part'] + $r['sen_part'];
    $r['overall']   = $r['tot_comp'] + $r['tot_part'];
  } unset($r);

  // --- 6) Dense ranks per block & overall ---
  $dense_rank = function(array &$arr, string $field, string $out){
    $copy = array_values($arr);
    usort($copy, function($a,$b) use ($field){
      if ($a[$field] == $b[$field]) return 0;
      return ($a[$field] > $b[$field]) ? -1 : 1;
    });
    $rank = 0; $prev = null; $pos = 0; $map = [];
    foreach ($copy as $row) {
      $pos++; $val = $row[$field];
      if ($prev === null || $val !== $prev) { $rank = $pos; $prev = $val; }
      $map[$val] = $rank;
    }
    foreach ($arr as &$row) $row[$out] = $map[$row[$field]] ?? null;
  };
  $dense_rank($rows, 'jun_total','jun_rank');
  $dense_rank($rows, 'int_total','int_rank');
  $dense_rank($rows, 'sen_total','sen_rank');
  $dense_rank($rows, 'overall', 'overall_rank');

  // Output sorted list
  $list = array_values($rows);
  usort($list, function($a,$b){
    if ($a['overall'] == $b['overall']) return 0;
    return ($a['overall'] > $b['overall']) ? -1 : 1;
  });

  json_ok(['rows'=>$list]);
}


/* ================================================================
   Unknown action
================================================================ */
json_err('unknown action', 400);
