<?php
/**
 * lib/scoring.php
 * Scoring & leaderboard helpers for TBC Athletics
 *
 * Depends on:
 *   - config.php  (DB + scoring constants)
 *   - lib/db.php  (get_pdo)
 */

require_once __DIR__ . '/db.php';
require_once __DIR__ . '/../config.php';

/**
 * Basic helpers
 */
function get_houses(PDO $pdo): array {
  return $pdo->query("SELECT house_code, house_name FROM houses ORDER BY house_code")->fetchAll();
}

/**
 * Competitive points per house (10/6/4) from:
 *  - Individual results: student's house (via students table)
 *  - Team results (e.g., 4x100): result's house_code
 */
function competitive_points_by_house(PDO $pdo): array {
  global $COMPETITIVE_POINTS;

  // Individual (student-based)
  $stmt = $pdo->prepare("
    SELECT s.house_code AS house_code,
           SUM(CASE r.place WHEN 1 THEN :p1 WHEN 2 THEN :p2 WHEN 3 THEN :p3 ELSE 0 END) AS points
    FROM competitive_results r
    JOIN students s       ON s.id = r.student_id
    JOIN scheduled_events se ON se.id = r.sched_id
    WHERE se.division='Competitive' AND r.status='OK'
    GROUP BY s.house_code
  ");
  $stmt->execute([
    ':p1' => (int)$COMPETITIVE_POINTS[1],
    ':p2' => (int)$COMPETITIVE_POINTS[2],
    ':p3' => (int)$COMPETITIVE_POINTS[3],
  ]);
  $ind = $stmt->fetchAll();

  // Team (house-based)
  $stmt2 = $pdo->prepare("
    SELECT r.house_code AS house_code,
           SUM(CASE r.place WHEN 1 THEN :p1 WHEN 2 THEN :p2 WHEN 3 THEN :p3 ELSE 0 END) AS points
    FROM competitive_results r
    JOIN scheduled_events se ON se.id=r.sched_id
    WHERE se.division='Competitive' AND r.status='OK' AND r.house_code IS NOT NULL
    GROUP BY r.house_code
  ");
  $stmt2->execute([
    ':p1' => (int)$COMPETITIVE_POINTS[1],
    ':p2' => (int)$COMPETITIVE_POINTS[2],
    ':p3' => (int)$COMPETITIVE_POINTS[3],
  ]);
  $team = $stmt2->fetchAll();

  // Aggregate
  $agg = [];
  foreach ($ind as $r)  { $agg[$r['house_code']] = ($agg[$r['house_code']] ?? 0) + (int)$r['points']; }
  foreach ($team as $r) { $agg[$r['house_code']] = ($agg[$r['house_code']] ?? 0) + (int)$r['points']; }

  $rows = [];
  foreach ($agg as $h => $p) { $rows[] = ['house_code' => $h, 'points' => (int)$p]; }
  return $rows;
}

/**
 * Participation points per house.
 * Entrants = PARTICIPATION_ENTRANT_POINTS each
 * Finalists = PARTICIPATION_FINALIST_POINTS each (either replaces entrant points or adds on top,
 * controlled by PARTICIPATION_FINALISTS_ARE_ADDITIONAL)
 */
function participation_points_by_house(PDO $pdo): array {
  $entrantPts  = (int)PARTICIPATION_ENTRANT_POINTS;
  $finalistPts = (int)PARTICIPATION_FINALIST_POINTS;
  $additional  = (bool)PARTICIPATION_FINALISTS_ARE_ADDITIONAL;

  // Entrants
  $stmt = $pdo->prepare("
    SELECT s.house_code, COUNT(*) * :e AS points
    FROM participation_entries pe
    JOIN students s ON s.id = pe.student_id
    GROUP BY s.house_code
  ");
  $stmt->execute([':e' => $entrantPts]);
  $ent = $stmt->fetchAll();

  $agg = [];
  foreach ($ent as $r) { $agg[$r['house_code']] = (int)$r['points']; }

  // Finalists
  $fin = $pdo->query("
    SELECT s.house_code, COUNT(*) AS finalists
    FROM participation_entries pe
    JOIN students s ON s.id=pe.student_id
    WHERE pe.finalist=1
    GROUP BY s.house_code
  ")->fetchAll();

  foreach ($fin as $r) {
    $h   = $r['house_code'];
    $cnt = (int)$r['finalists'];
    if ($additional) {
      $agg[$h] = ($agg[$h] ?? 0) + ($cnt * $finalistPts);
    } else {
      // finalists should total 'finalistPts' (replace entrant point with finalist point)
      $agg[$h] = ($agg[$h] ?? 0) + ($cnt * ($finalistPts - $entrantPts));
    }
  }

  $rows = [];
  foreach ($agg as $h => $p) { $rows[] = ['house_code' => $h, 'points' => (int)$p]; }
  return $rows;
}

/**
 * Overall house totals: competitive + participation
 */
function get_house_totals(PDO $pdo): array {
  $houses = get_houses($pdo);
  $map = [];
  foreach ($houses as $h) {
    $map[$h['house_code']] = [
      'house_name'    => $h['house_name'],
      'competitive'   => 0,
      'participation' => 0,
      'total'         => 0,
    ];
  }

  foreach (competitive_points_by_house($pdo) as $r) {
    if (!isset($map[$r['house_code']])) continue;
    $map[$r['house_code']]['competitive'] = (int)$r['points'];
  }
  foreach (participation_points_by_house($pdo) as $r) {
    if (!isset($map[$r['house_code']])) continue;
    $map[$r['house_code']]['participation'] = (int)$r['points'];
  }
  foreach ($map as $k => $v) {
    $map[$k]['total'] = (int)$v['competitive'] + (int)$v['participation'];
  }

  // Sort by total desc
  uasort($map, function($a,$b){ return $b['total'] <=> $a['total']; });
  return $map; // associative keyed by house_code
}

/* =============================================================================
 * Banded leaderboard (Junior / Inter. / Senior + Totals)
 * ========================================================================== */

/** Junior = Y9–10, Inter. = Y11, Senior = Y12–13 */
function tbc_band_from_year(int $y): string {
  if ($y === 9 || $y === 10) return 'junior';
  if ($y === 11)             return 'inter';
  if ($y === 12 || $y === 13) return 'senior';
  return 'other';
}

/** Deduce band from a scheduled_event.year_list like "9,10" or "11,12,13" */
function tbc_band_from_yearlist(?string $yl): string {
  if (!$yl) return 'other';
  $yl = ','.preg_replace('/\s+/', '', $yl).','; // ,9,10, or ,11,12,
  if (preg_match('/,(9|10),/',  $yl)) return 'junior';
  if (preg_match('/,(11),/',    $yl)) return 'inter';
  if (preg_match('/,(12|13),/', $yl)) return 'senior';
  return 'other';
}

/**
 * Full per-house breakdown:
 *  - junior/inter/senior: comp, part, total, rank
 *  - totals: comp, part, overall, rank
 *
 * Returns a numerically indexed array of rows sorted by overall desc:
 * [
 *   [
 *     'house_code','house_name',
 *     'jun_comp','jun_part','jun_total','jun_rank',
 *     'int_comp','int_part','int_total','int_rank',
 *     'sen_comp','sen_part','sen_total','sen_rank',
 *     'tot_comp','tot_part','overall','overall_rank'
 *   ], ...
 * ]
 */
function get_house_breakdown(PDO $pdo): array {
  global $COMPETITIVE_POINTS;
  $p1 = (int)$COMPETITIVE_POINTS[1];
  $p2 = (int)$COMPETITIVE_POINTS[2];
  $p3 = (int)$COMPETITIVE_POINTS[3];
  $entrantPts  = (int)PARTICIPATION_ENTRANT_POINTS;
  $finalistPts = (int)PARTICIPATION_FINALIST_POINTS;
  $additional  = (bool)PARTICIPATION_FINALISTS_ARE_ADDITIONAL;

  // Seed houses
  $H = [];
  foreach ($pdo->query("SELECT house_code, house_name FROM houses ORDER BY house_code") as $h) {
    $H[$h['house_code']] = [
      'house_code' => $h['house_code'],
      'house_name' => $h['house_name'],
      'junior' => ['comp'=>0,'part'=>0,'total'=>0,'rank'=>0],
      'inter'  => ['comp'=>0,'part'=>0,'total'=>0,'rank'=>0],
      'senior' => ['comp'=>0,'part'=>0,'total'=>0,'rank'=>0],
      'totals' => ['comp'=>0,'part'=>0,'overall'=>0,'rank'=>0],
    ];
  }

  // Competitive (individual)
  $sqlInd = "
    SELECT s.house_code h, s.year_level y,
           SUM(CASE r.place WHEN 1 THEN :p1 WHEN 2 THEN :p2 WHEN 3 THEN :p3 ELSE 0 END) pts
    FROM competitive_results r
    JOIN students s ON s.id=r.student_id
    JOIN scheduled_events se ON se.id=r.sched_id
    WHERE se.division='Competitive' AND r.status='OK'
    GROUP BY s.house_code, s.year_level
  ";
  $st = $pdo->prepare($sqlInd);
  $st->execute([':p1'=>$p1, ':p2'=>$p2, ':p3'=>$p3]);
  foreach ($st->fetchAll() as $r) {
    if (!isset($H[$r['h']])) continue;
    $band = tbc_band_from_year((int)$r['y']);
    $H[$r['h']][$band]['comp'] += (int)$r['pts'];
  }

  // Competitive (team/relay via house_code + sched.year_list)
  $sqlTeam = "
    SELECT r.house_code h, se.year_list yl,
           SUM(CASE r.place WHEN 1 THEN :p1 WHEN 2 THEN :p2 WHEN 3 THEN :p3 ELSE 0 END) pts
    FROM competitive_results r
    JOIN scheduled_events se ON se.id=r.sched_id
    WHERE se.division='Competitive' AND r.status='OK' AND r.house_code IS NOT NULL
    GROUP BY r.house_code, se.year_list
  ";
  $st = $pdo->prepare($sqlTeam);
  $st->execute([':p1'=>$p1, ':p2'=>$p2, ':p3'=>$p3]);
  foreach ($st->fetchAll() as $r) {
    if (!isset($H[$r['h']])) continue;
    $band = tbc_band_from_yearlist($r['yl']);
    $H[$r['h']][$band]['comp'] += (int)$r['pts'];
  }

  // Participation entrants
  foreach ($pdo->query("
    SELECT s.house_code h, s.year_level y, COUNT(*) cnt
    FROM participation_entries pe
    JOIN students s ON s.id=pe.student_id
    GROUP BY s.house_code, s.year_level
  ")->fetchAll() as $r) {
    if (!isset($H[$r['h']])) continue;
    $band = tbc_band_from_year((int)$r['y']);
    $H[$r['h']][$band]['part'] += (int)$r['cnt'] * $entrantPts;
  }

  // Participation finalists
  foreach ($pdo->query("
    SELECT s.house_code h, s.year_level y, COUNT(*) finalists
    FROM participation_entries pe
    JOIN students s ON s.id=pe.student_id
    WHERE pe.finalist=1
    GROUP BY s.house_code, s.year_level
  ")->fetchAll() as $r) {
    if (!isset($H[$r['h']])) continue;
    $band = tbc_band_from_year((int)$r['y']);
    $cnt  = (int)$r['finalists'];
    $H[$r['h']][$band]['part'] += $additional ? ($cnt * $finalistPts)
                                              : ($cnt * ($finalistPts - $entrantPts));
  }

  // Totals per band + grand totals
  foreach ($H as $code => $row) {
    foreach (['junior','inter','senior'] as $b) {
      $H[$code][$b]['total'] = (int)$H[$code][$b]['comp'] + (int)$H[$code][$b]['part'];
      $H[$code]['totals']['comp'] += (int)$H[$code][$b]['comp'];
      $H[$code]['totals']['part'] += (int)$H[$code][$b]['part'];
    }
    $H[$code]['totals']['overall'] = (int)$H[$code]['totals']['comp'] + (int)$H[$code]['totals']['part'];
  }

  // Dense rank helper
  $rankBand = function(array $H, string $field, string $sumKey) {
    $list = [];
    foreach ($H as $c => $r) {
      $list[] = ['c' => $c, 'v' => $field === 'totals' ? $r['totals'][$sumKey] : $r[$field][$sumKey]];
    }
    usort($list, function($a,$b){ return $b['v'] <=> $a['v']; });
    $rank = 0; $prev = null; $seen = 0;
    foreach ($list as $item) {
      $seen++;
      if ($prev === null || $item['v'] !== $prev) { $rank = $seen; $prev = $item['v']; }
      if ($field === 'totals') $H[$item['c']]['totals']['rank'] = $rank;
      else                     $H[$item['c']][$field]['rank']   = $rank;
    }
    return $H;
  };

  $H = $rankBand($H, 'junior', 'total');
  $H = $rankBand($H, 'inter',  'total');
  $H = $rankBand($H, 'senior', 'total');
  $H = $rankBand($H, 'totals', 'overall');

  // Flatten for API consumers
  $out = [];
  foreach ($H as $c => $r) {
    $out[] = [
      'house_code'  => $c,
      'house_name'  => $r['house_name'],
      'jun_comp'    => (int)$r['junior']['comp'],
      'jun_part'    => (int)$r['junior']['part'],
      'jun_total'   => (int)$r['junior']['total'],
      'jun_rank'    => (int)$r['junior']['rank'],
      'int_comp'    => (int)$r['inter']['comp'],
      'int_part'    => (int)$r['inter']['part'],
      'int_total'   => (int)$r['inter']['total'],
      'int_rank'    => (int)$r['inter']['rank'],
      'sen_comp'    => (int)$r['senior']['comp'],
      'sen_part'    => (int)$r['senior']['part'],
      'sen_total'   => (int)$r['senior']['total'],
      'sen_rank'    => (int)$r['senior']['rank'],
      'tot_comp'    => (int)$r['totals']['comp'],
      'tot_part'    => (int)$r['totals']['part'],
      'overall'     => (int)$r['totals']['overall'],
      'overall_rank'=> (int)$r['totals']['rank'],
    ];
  }
  usort($out, function($a,$b){ return $b['overall'] <=> $a['overall']; });
  return $out;
}
